/************************************************************************************************************************************************
** DT9062 Arduino-Library
** Einlesen des Displays eines DT9062-Multimeters
** ber die serielle Schnittstelle in einen Arduino
** Sowohl HardwareSerial- als auch SoftwareSerial-Ports knnen benutzt werden.
** Version: 1.0

** Software License Agreement
** Copyright (c) 2020 by Klaus Rittmeier
** All rights reserved.

** Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

**  Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
**  Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
**  Neither the name of the copyright owners nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
** INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
** OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
** OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*************************************************************************************************************************************************/

#include <Dt9062.h>

/***************************************************************************************
** public
****************************************************************************************/
Dt9062::Dt9062()
{
  bufferindex=0;
  pSerial=NULL;
}

void Dt9062::connect(Stream &serial)
{
	pSerial = &serial;
}

/***************************************************************************************
** available() wird in der zentralen loop()-Funktion zyklisch aufgerufen.
****************************************************************************************/
bool Dt9062::available()
{
  if(pSerial != NULL && pSerial->available()>0)
  {
    unsigned char in = pSerial->read();
	if((in & 0xF0) == 0x10)
      bufferindex=0;
    buffer[bufferindex++] = in;
    if(bufferindex>=14)
    {
      bufferindex=0;
      setInfo();
      setValue();
      setUnit();
      setCurrent();
      return true;
    }
  }
  return false;
}

/***************************************************************************************
** private
****************************************************************************************/
//--------------------------------------------------------------------
// Erzeugt aus 3 niederwertigen Bits vom ersten Byte und 4 niederwertigen Bits
// vom zweiten Byte einen 7-Bit-Wert, der einer Ziffer zugeordnet wird.
//            Bit0(1)
//           XXX 1 XXX
//          X         X
// Bit1(2)  6         2  Bit3(8)
//          X   Bit4  X
//           XXX 7 XXX
//          X   (16)  X
// Bit2(4)  5         3  Bit5(32)
//          X         X
//           XXX 4 XXX
//            Bit6(64)
int Dt9062::ziffer(unsigned char byte1, unsigned char byte2)
{
  int siebenbit, ziffer;

  siebenbit = (byte1 & 0x07) | (8 * (byte2 & 0x0F));
  switch(siebenbit)
  {
    case  40: ziffer=1; break;  // 00101000b
    case  93: ziffer=2; break;  // 01011101b
    case 121: ziffer=3; break;
    case  58: ziffer=4; break;
    case 115: ziffer=5; break;
    case 119: ziffer=6; break;
    case  41: ziffer=7; break;
    case 127: ziffer=8; break;
    case 123: ziffer=9; break;
    default:  ziffer=0;
  }
  return ziffer;
}

/*********************************************************************************
 * In den folgenden Funktionen wurden bewusst keine str...()-Funktionen verwendet.
 * Das spart RAM und Laufzeit.
 * ******************************************************************************/
// wertet die Statuszeile der Anzeige aus
void Dt9062::setInfo()
{
  int i = 0;

  if(buffer[0] & 2)
    { _info[i++]=' '; _info[i++]='A'; _info[i++]='u'; _info[i++]='t'; _info[i++]='o'; }
  if(buffer[0] & 1)
    { _info[i++]=' '; _info[i++]='R'; _info[i++]='S'; _info[i++]='2'; _info[i++]='3'; _info[i++]='2'; }
  if(buffer[9] & 1)
    { _info[i++]=' '; _info[i++]='D'; _info[i++]='i'; _info[i++]='o'; _info[i++]='d'; _info[i++]='e'; }
  if(buffer[10] & 1)
    { _info[i++]=' '; _info[i++]='T'; _info[i++]='o'; _info[i++]='n'; }
  if (buffer[11] & 1)
    { _info[i++]=' '; _info[i++]='h'; _info[i++]='o'; _info[i++]='l'; _info[i++]='d'; }
  if(buffer[11] & 2)
    { _info[i++]=' '; _info[i++]='R'; _info[i++]='e'; _info[i++]='l'; }
  if(buffer[12] & 1)
    { _info[i++]=' '; _info[i++]='B'; _info[i++]='a'; _info[i++]='t'; _info[i++]='t'; }

  _info[i] = '\0';
}

// Bildet aus 4 Bytepaaren die Anzeigezahl und wertet Komma- und Vorzeichenbits aus.
void Dt9062::setValue()
{
  int zahl;

  zahl  = ziffer(buffer[1], buffer[2]) * 1000;
  zahl += ziffer(buffer[3], buffer[4]) * 100;
  zahl += ziffer(buffer[5], buffer[6]) * 10;
  zahl += ziffer(buffer[7], buffer[8]);
  _value = zahl;
// Kommastelle
  if(buffer[7] & 8)
    _value /= 10;
  else if(buffer[5] & 8)
    _value /= 100;
  else if(buffer[3] & 8)
    _value /= 1000;
// Vorzeichen
  if(buffer[1] & 8)
    _value = -_value;
}

// AC oder DC
void Dt9062::setCurrent()
{
  _current[0]='\0', _current[1]='C', _current[2]='\0';
  if(buffer[0] & 4)
    _current[0]='D';
  if(buffer[0] & 8)
    _current[0]='A';
}

// Maeinheit bestimmen
void Dt9062::setUnit()
{
  int  i=0;

  if(buffer[9] & 2)
    _unit[i++]='k';
  else if(buffer[9] & 4)
    _unit[i++]='n';
  else if(buffer[9] & 8)
    { _unit[i++]='m'; _unit[i++]='y'; }
  else if(buffer[10] & 2)
    _unit[i++]='M';
  else if(buffer[10] & 4)
    _unit[i++]='%';
  else if(buffer[10] & 8)
    _unit[i++]='m';

  if(buffer[11] & 4)
    { _unit[i++]='O'; _unit[i++]='h'; _unit[i++]='m'; }
  else if(buffer[11] & 8)
    _unit[i++]='F';
  else if(buffer[12] & 2)
    { _unit[i++]='H'; _unit[i++]='z'; }
  else if(buffer[12] & 4)
    _unit[i++]='V';
  else if(buffer[12] & 8)
    _unit[i++]='A';
  else if(buffer[13] & 2)
    { _unit[i++]='H'; _unit[i++]='F'; _unit[i++]='E'; }
  else if(buffer[13] & 4)
    { _unit[i++]='\337'; _unit[i++]='C'; }

  _unit[i]=0;
}
